/* atx1.c  an assembler for tx1

format  (case insensitive)

	.symbol                   define symbol
	symbol value
	 ...
	.code ads                 code segment at ads
	[:label] opcode operand
	 ...
	.data ads				  data segment at ads
	n n ...                   define words
	.end

operand
L-format: op ads
R-format: op xop r2
I-format: op xop im

23 june 2017

Prabhas Chongstitvatana
*/

#include "atx1.h"

char 	lbuf[MXBUF];				// copy of input
char 	ibuf[MXBUF], *cp = NULL;   	// input buffer, ch ptr
char	ob[MXBUF];					// output buffer
char	cbuf[MXBUF];				// output buffer for listing

static 	char sep[] = " \t\n";      	// separator char
char    *w;							// current input word
int 	lineno,loc;		   			// line num, current ads
FILE 	*fi, *fo, *fl;
token	mem[MXMEM];					// store tokens for pass2
int 	tp = 0;						// token index
//int 	mark;						// index to current op token
int		pass;
int		numop;						// number of initial symbols (op)

// initial symbol in the symbol table

typedef struct { char name[8]; int value; } symtype;

symtype	initsym[] = {
	{"NOP",0},{"LDA",LDA},{"STA",STA},
	{"JMP",JMP},{"JT",JT},{"JF",JF},{"CALL",CALL},
	{"ADD",ADD},{"SUB",SUB},
	{"AND",AND},{"OR",OR},{"XOR",XOR},
	{"EQ",EQ},{"LT",LT}, {"LE",LE},
	{"GT",GT},{"GE",GE},
	{"ADDI",ADDI},{"SUBI",SUBI},
	{"ANDI",ANDI},{"ORI",ORI},{"XORI",XORI},
	{"EQI",EQI},{"LTI",LTI}, {"LEI",LEI},
	{"GTI",GTI},{"GEI",GEI},
	{"MVA",MVA},{"MVR",MVR},{"LDX",LDX},{"STX",STX},
	{"MVI",MVI},{"RET",RET},{"NOT",NOT},
	{"TRAP",TRAP},{"CLR",CLR},{"INC",INC},{"DEC",DEC},

	{"R0",0},{"R1",1},{"R2",2},{"R3",3},{"R4",4},
	{"R5",5},{"R6",6},{"R7",7},{"R8",8},{"R9",9},
	{"R10",10},{"R11",11},{"R12",12},{"R13",13},{"R14",14},
	{"R15",15},{"BP",15},
	{"",0}
};

void error(char *s){
	if(pass == 1)
		printf("pass 1 line %d error: %s symbol %s\n%s",lineno,s,w,lbuf);
	else
		printf("pass 2 line %d error: %s\n",mem[tp].line,s);
	exit(0);
}
// get one token from input
char *tok(void){
	if( cp != NULL ) cp = strtok(NULL,sep);
	while ( cp == NULL || eqs(cp,";") ){
		if( fgets(ibuf,MXBUF,fi) == NULL ) return NULL;
		strcpy(lbuf,ibuf);
		lineno++;
		cp = strtok(ibuf, sep);
	}
	cp = strupr(cp);   // convert to uppercase
	return cp;
}
// store token
void store(char type, int ref){
	if( tp >= MXMEM ) error("out of memory");
	mem[tp].type = type;
//	mem[tp].mode = mode;
	mem[tp].ref = ref;
	mem[tp].line = lineno;
	tp++;
}

// check number
int isnumber(char *s){
	int i = 0;
	if(s[0] == '-') i = 1;
	while( isdigit(s[i]) ) i++;
	if( s[i] == 0 ) return 1;
	return 0;
}

int getNum(void){
	w = tok();
	if(isnumber(w))	return atoi(w);
	error("expect number");
	return 0;
}

int valueof(char *s, int *type){
	*type = NUM;
	if( isnumber(s) ) return atoi(s);
	*type = SYM;
	return putsym(s);
}

// read reg, mode RR
void readR(void){
	int v,type;
	w = tok();
	v = valueof(w,&type);
	store(type,v);
}

void storew(char *s){
	int v,type;
	v = valueof(s,&type);
	store(type,v);
}
// read mem, set mode ABS IND INX
void readM(void){
	w = tok();
	storew(w);
}

// read imm
void readI(void){
	w =tok();
	if( w[0] == '#' ) storew(w+1);
	else  error("expect immediate argument");
}

// read argument according to OP
void readcode(int op){

	store(OP,op);
	switch( op ) {

	case LDA:	// ads arg
	case STA:
	case JMP:
	case JT:
	case JF:
	case CALL:  readM(); break;

	case ADD:	// reg arg
	case SUB:
	case AND:
	case OR:
	case XOR:
	case EQ:
	case LT:
	case LE:
	case GT:
	case GE:
	case MVA:
	case MVR:
	case LDX:
	case STX:
	case CLR:
	case INC:
	case DEC:  readR(); break;

	case ADDI:	// imm arg
	case SUBI:
	case ANDI:
	case ORI:
	case XORI:
	case EQI:
	case LTI:
	case LEI:
	case GTI:
	case GEI:
	case MVI:
	case TRAP:  readI(); break;

	case RET:		// zero arg
	case NOT: break;

	default: error("undefine op");
	}
}

void pass1(void){
	int idx, state;
	pass = 1;
	state = 'S';
	w = tok();
	while( !eqs(w,".END") ) {
		if( eqs(w,".SYMBOL")){
			state = 'S';
		}else if(eqs(w,".DATA")){
			loc = getNum();
			store(DOTD,loc);
			state = 'D';
		}else if(eqs(w,".CODE")){
			loc = getNum();
			store(DOTC,loc);
			state = 'C';
		}else
		switch(state){
		case 'S':
			idx = putsym(w);
			if(getValue(idx) != UNDEF)
				error("duplicate symbol");
			setsym(idx,getNum(),SYM,0);
			break;
		case 'D':
			if( isnumber(w) )
				store(NUM,atoi(w));
			else
				store(SYM,putsym(w));
			loc++;
			break;
		case 'C':
			if( w[0] == ':' ){	// insert symbol
				idx = putsym(w+1);
				if(getValue(idx) != UNDEF)
					error("duplicate label");
				setsym(idx,loc,SYM,0);
			}else{
				idx = searchsym(w);
				if(idx < 0 || getType(idx) != OP)
					error("undefined op");
				readcode(getValue(idx));
				loc++;
			}
		} // switch
		w = tok();
	}  // while
	store(DOTE,loc);
}

int isdot(char type){
	return type == DOTC || type == DOTD || type == DOTE;
}

int rdtokval(void){
	int v;
	v = UNDEF;
	switch( mem[tp].type ){
	case NUM: v = mem[tp].ref; break;
	case SYM: v = getValue(mem[tp].ref);
	}
	if(v == UNDEF) error("undefined symbol");
	tp++;
	return v;
}

PRIVATE void prL(int op, int a1){
	sprintf(ob,"L %d %d",op,a1);
}
PRIVATE void prR(int op, int a1){
	sprintf(ob,"R 14 %d %d",op,a1);
}
PRIVATE void prI(int op, int a1){
	sprintf(ob,"I 15 %d %d",op,a1);
}

void gencode(void){
	int op, a1;

	op = mem[tp].ref;
	tp++;
	switch( op ){
	case NOP: prL(0,0); break;
	case LDA:
	case STA:
	case JMP:
	case JT:
	case JF:
	case CALL: prL(op,rdtokval()); break;

	case ADD:
	case SUB:
	case AND:
	case OR:
	case XOR:
	case EQ:
	case LT:
	case LE:
	case GT:
	case GE: prR(op-10,rdtokval()); break;

	case MVA:
	case MVR:
	case LDX:
	case STX: prR(op-20,rdtokval()); break;

	case ADDI:
	case SUBI:
	case ANDI:
	case ORI:
	case XORI:
	case EQI:
	case LTI:
	case LEI:
	case GTI:
	case GEI: prI(op-20,rdtokval()); break;

	case MVI:
	case TRAP: prI(op-24,rdtokval()); break;
	case RET: sprintf(ob,"I 15 12 0"); break;
	case NOT: sprintf(ob,"I 15 13 0"); break;
	case CLR: sprintf(ob,"R 13 0 %d",rdtokval()); break;
	case INC: sprintf(ob,"R 13 1 %d",rdtokval()); break;
	case DEC: sprintf(ob,"R 13 2 %d",rdtokval()); break;

	default: error("undefine op");
	}
}

void outobj(void){
	fprintf(fo,"%s\n",ob);
}

void outlst(void){
	int i, line;
	line = mem[tp-1].line;
	// get source until before current line
	while(lineno < (line-1)){
		fgets(lbuf,MXBUF,fi);
		fprintf(fl,"                        %s",lbuf);
		lineno++;
	}
	fgets(lbuf,MXBUF,fi);	// current line
	lineno++;
	sprintf(cbuf,"%4d %s",loc,ob);
	for(i=strlen(cbuf); i<24; i++)	// pad blank to 24
		cbuf[i] = 32;
	strcpy(cbuf+24,lbuf);
	fprintf(fl,"%s",cbuf);
}

void pass2(void){
	pass = 2;
	tp = 0;
	lineno = 0;
	rewind(fi);
	while( mem[tp].type != DOTE){
		switch(mem[tp].type){
		case DOTC:
			loc = mem[tp].ref;
			sprintf(ob,"a %d",loc);
			outobj();
			tp++;
			while(!isdot(mem[tp].type)){
				gencode();
				outobj();
//				outlst();
				loc++;
			}
			break;
		case DOTD:
			loc = mem[tp].ref;
			sprintf(ob,"a %d",loc);
			outobj();
			tp++;
			while(!isdot(mem[tp].type)){
				sprintf(ob,"w %d",rdtokval());
				outobj();
				loc++;
			}
		}
	}
	sprintf(ob,"e");
	outobj();
	// list rest of the source
//	while(fgets(lbuf,MXBUF,fi) != NULL)
//		fprintf(fl,"                        %s",lbuf);
}

// put reserved words into symbol table
void initsymbol(void){
	int i;
	for(i=0; initsym[i].name[0] != 0; i++)
		setsym(putsym(initsym[i].name),
			initsym[i].value,OP,0);
	numop = i;
}

// make an obj file name from source
void makename( char *source, char *obj, char *lis ){
	int n;
	n = strcspn(source,".");
	strncpy(obj,source,n);
	strcpy(obj+n,".obj");
	strncpy(lis,source,n);
	strcpy(lis+n,".lis");
}

void main(int argc, char *argv[]){
	char fout[80], flis[80];

	if( argc < 2 ) {
		printf("usage : as21 inputfile\n");
		exit(0);
	}
	fi = fopen(argv[1],"r");
	if( fi == NULL ){
		printf("input file not found\n");
		exit(0);
	}
	makename(argv[1], fout, flis);
	initsymbol();
	lineno = 0;
	loc = 0;
	pass1();
//	fl = fopen(flis,"w");
	fo = fopen(fout,"w");
	pass2();
//	dumpsymbol();
	fclose(fi);
	fclose(fo);
//	fclose(fl);
}

/*----------- for debugging


void testread(void){
	int cnt = 100;
	w = tok();
	while ( !eqs(w,".END") && (w != NULL) ){
		printf("%s ",w);
		w = tok();
		if(cnt-- < 0) break;
	}
}

char prmode( int mode){
	switch( mode ){
	case ABS: return 'A';
//	case DISP:return '@';
	case INX:return '+';
	case IMM:return '%';
	case RI:return '#';
	case RR:return 'R';
	case SP:return 'S';
	case NA:return '?';
//	case EL: return 'L';
//	case ED: return 'D';
//	case EX: return 'X';
	}
	return 0;
}

void prtoken(int type, int mode, int ref){
	switch( type ){
	case SYM: printf("sym %c %d\n",prmode(mode),ref); break;
	case NUM: printf("num %c %d\n",prmode(mode),ref); break;
	case OP: printf("op %c %d\n",prmode(mode),ref); break;
//	case DOTA: printf(".a %d\n",ref); break;
	case DOTC: printf(".c %d\n",ref); break;
	case DOTD: printf(".w %d\n",ref); break;
	case DOTE: printf(".e\n"); break;
	}
}

void dumptoken(){
	int i;
	for(i=0;i<tp;i++)
		prtoken(mem[i].type,mem[i].mode,mem[i].ref);
}

--------------- */
