
// mon.c  monitor for command line interpreter
//    s30 multicore              	17 Apr 2016

#include "s30.h"

#define MXBUF  80
#define MXOBUF 2000
#define MXBP  20
#define sep  "\t\n "

extern int NC;
extern int runflag[], isWait[];
extern int R[][32], Ir[], Pc[], M[];
extern int T;

int currentcore;					// focus core
int brkp = -1;						// breakpoint

char ibuf[MXBUF], *cp = NULL;		/* input buffer */
FILE *fi, *fo;
int disp = 1;						/* display control, default all cores */

PRIVATE char *tok(void){	/* get one token from input */
	if( cp != NULL ) cp = strtok(NULL,sep);
	while ( cp == NULL ){
		printf(">");
		cp = fgets(ibuf,MXBUF,fi);
		if( cp != NULL ) cp = strtok(ibuf, sep);
	}
	cp = strupr(cp);   /* convert to uppercase */
// printf("%s ",cp);
	return cp;
}

void showK(int k);

PRIVATE void show(void){
	int i;
	printf("T %d\n",T);
	if( disp ){
		for(i = 0; i < NC; i++)
			showK(i);
	}else
		showK(currentcore);
}

PRIVATE int notstop(int k){
	return runflag[k] || isWait[k];
}

//  if set "all" (disp == 1) then run until all cores stop
PRIVATE int chkstop(void){
	int i, flag;
	flag = 0;
	if( disp ){				// all cores
		for(i=0; i<NC; i++){
			if(Pc[i] == brkp){  // some core breakpoint
			    flag = 0;
				break;
			}
			if(notstop(i)){	   // some core still run
				flag = 1;
				break;
			}
		}
	}else{					// check current core
		i = currentcore;
		flag = Pc[i] != brkp && notstop(i);
	}
	return flag;
}

PRIVATE void trace(void){
	while( chkstop() ) {
		sim();
		if( T > XLIMIT ){
			printf("infinite loop\n");
			break;
		}
	}
}

PRIVATE int isReg(char *s){
	int n;
	if( s[0] == 'R') {
		n = atoi(s+1);
		if( n > 0 && n < 32 ) return 1;
	}
	return 0;
}

PRIVATE void setArg(char *s, int value){
	if( isReg(s) )
		R[currentcore][atoi(s+1)] = value;
	else if ( eqs(s,"PC") )
		Pc[currentcore] = value;
	else if ( s[0] == 'M' )
		M[atoi(s+1)] = value;
	else
		printf("set unknow argument\n");
}

PRIVATE void dumpMem(int ads, int n){
	int i,k;
	k = 0;
	printf("%4d : ",ads);
	for(i=ads;i<ads+n;i++){
		printf("%d ",M[i]);
		k++;
		if( k%10 == 0 ) printf("\n%4d : ",i+1);
	}
	printf("\n");
}

PRIVATE void help(void){
	printf("%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
	  "a - show all cores",
	  "g - go", "t - single step", "b ads - set breakpoint",
	  "c n - focus core n", "s [rn,mn,pc] v - set",
	  "d ads n - dump", "r - show register",
	  "q - quit", "h - help");
}

void interp(void){
	char *arg;
	int i, v, v2, runf;

	runf = 1;
	fi = stdin;
	while(runf){
		arg = tok();
		switch( arg[0] ){
		case 'A': 			// show all cores
			disp = 1;
			printf("mode show all cores\n");
			break;
		case 'G': trace(); break;
		case 'T': sim(); show(); break;
		case 'B': brkp = atoi(tok()); break;
		case 'C':
			v = atoi(tok());
			if( v > NC ) v = NC-1;
			currentcore = v;
			disp = 0;
			break;
		case 'S':
			arg = tok();
			v = atoi(tok());
			setArg(arg,v);
			break;
		case 'D':
			v = atoi(tok());
			v2 = atoi(tok());
			dumpMem(v,v2);
			break;

		case 'R': dumpreg(); break;
		case 'Q': runf = 0; break;
		case 'H': help(); break;
		default: printf("unknow command\n");
		}
	}
}

PRIVATE void allreg(int k){
	int i;
	for(i=0; i<32; i++){
	   	if(i%8 == 0) printf("\n");
	   	printf("r%-2d:%-4d ",i,R[k][i]);
	}
}

void dumpreg(void){
	int k;
	if( disp ){
		for(k=0; k<NC; k++){
			printf("\ncore %d",k);
			allreg(k);
		}
	}else
		allreg(currentcore);
	printf("\n");
}

/*
void testtok(){
	char *w;
	fi = stdin;
	cp = NULL;
	while(1){
		w = tok();
		printf("%s ",w);
		if( eqs(w,"QUIT")) break;
	}
}
*/
