
/* S3 multicore simulator

   S2.1 instruction set with interrupt
   first release						17 Apr 2016
   add: wfi, intx, sync, semaphore		20 Apr 2016
   update fix bugs sigx,ei,di		    28 Oct 2017

   P. Chongstitvatana
   Department of Computer Engineering
   Chulalongkorn University

*/

#include "s30.h"

int NC;					// number of core
int R[CX][32], Ir[CX], Pc[CX], M[MAXMEM];
int runflag[CX], savePc[CX], lastInt[CX], isWait[CX];
int T;					// global clock
int intflag[CX], intmask[CX][4], intrq[CX][4];

int signx2( int d ){	// sign bit 21 extended
	if( d & 0x0200000 ) return d | 0xFFC00000;
	return d;
}

// int #n at core #k
void interrupt(int k, int n){
	if( intflag[k] ){
		printf("interrupt core %d\n",k);
		R[k][31] = Pc[k];		// save pc
		Pc[k] = M[INTVEC+(NC*k+n)];
	}
}

void trap( int k, int reg, int num ){	// special function
	int a, i;

	switch( num ) {
	case 0:  // stop
		printf("core %d stop, clock %d\n",k,T);
		runflag[k] = 0;
		break;
	case 1: printf("%d ",R[k][reg]); break;			// print
	case 2: printf("%c",R[k][reg]); break;			// printc
/*
	case 15: intflag[k] = 0; break;			// di
	case 16: intflag[k] = 1; break;			// ei

	case 20: R[k][27] = newsem(); break;
	case 21: wait(R[k][1]); break;
	case 22: signal(R[k][1]); break;
*/
	}
}

PRIVATE void allsync(int k){
	int a, i;
	runflag[k] = 0;			// stop this core
	a = 0;
	for(i=0; i<NC; i++)		// check all cores
		if(runflag[i] == 0) a++;
	if(a == NC)				// all cores have stop
		for(i=0; i<NC; i++)
			runflag[i] = 1;
}

// *** how to set interrupt mask?   6 nov 2017
void checkinterrupt(int k){
	if( intflag[k] ){
		if( intmask[k][0] && intrq[k][0] ) interrupt(k,0);
		else if( intmask[k][1] && intrq[k][1] ) interrupt(k,1);
		else if( intmask[k][2] && intrq[k][2] ) interrupt(k,2);
		else if( intmask[k][3] && intrq[k][3] ) interrupt(k,3);
	}
}

void run(int k){		// execute one instruction
	int ads, d, r1, r2, r3, i;

	savePc[k] = Pc[k];
	Ir[k] = M[Pc[k]];		// instruction fetch
	Pc[k]++;
	ads = IRads(k);			// decode
	d = IRdisp(k);			// signx  17-bit
	r1 = IRr1(k);
	r2 = IRr2(k);
	r3 = IRr3(k);
	switch( IRop(k) ) {
	case NOP:  break;
	case LDA:  R[k][r1] = M[ads]; break;
	case LDD:  R[k][r1] = M[R[k][r2]+d]; break;
	case STA:  M[ads] = R[k][r1]; break;
	case STD:  M[R[k][r2]+d] = R[k][r1]; break;
	case MVI:  R[k][r1] = signx2(ads); break;
	case JMP:  Pc[k] = ads; break;
	case JAL:  R[k][r1] = Pc[k]; Pc[k] = ads; break;
	case JT: if( R[k][r1] != 0 ) Pc[k] = ads; break;
	case JF: if( R[k][r1] == 0 ) Pc[k] = ads; break;
	case ADDI: R[k][r1] = R[k][r2] + d; break;
	case SUBI: R[k][r1] = R[k][r2] - d; break;
	case MULI: R[k][r1] = R[k][r2] * d; break;
	case DIVI: R[k][r1] = R[k][r2] / d; break;
	case ANDI: R[k][r1] = R[k][r2] & d; break;
	case ORI:  R[k][r1] = R[k][r2] | d; break;
	case XORI: R[k][r1] = R[k][r2] ^ d; break;
	case EQI:  R[k][r1] = R[k][r2] == d; break;
	case NEI:  R[k][r1] = R[k][r2] != d; break;
	case LTI:  R[k][r1] = R[k][r2] <  d; break;
	case LEI:  R[k][r1] = R[k][r2] <= d; break;
	case GTI:  R[k][r1] = R[k][r2] >  d; break;
	case GEI:  R[k][r1] = R[k][r2] >= d; break;
	case SHLI: R[k][r1] = R[k][r2] << d; break;
	case SHRI: R[k][r1] = R[k][r2] >> d; break;
	case XOP:
		switch( IRxop(k) ){
		case ADD:  R[k][r1] = R[k][r2] + R[k][r3]; break;
		case SUB:  R[k][r1] = R[k][r2] - R[k][r3]; break;
		case MUL:  R[k][r1] = R[k][r2] * R[k][r3]; break;
		case DIV:  R[k][r1] = R[k][r2] / R[k][r3]; break;
		case AND:  R[k][r1] = R[k][r2] & R[k][r3]; break;
		case OR:   R[k][r1] = R[k][r2] | R[k][r3]; break;
		case XOR:  R[k][r1] = R[k][r2] ^ R[k][r3]; break;
		case EQ:   R[k][r1] = R[k][r2] == R[k][r3]; break;
		case NE:   R[k][r1] = R[k][r2] != R[k][r3]; break;
		case LT:   R[k][r1] = R[k][r2] <  R[k][r3]; break;
		case LE:   R[k][r1] = R[k][r2] <= R[k][r3]; break;
		case GT:   R[k][r1] = R[k][r2] >  R[k][r3]; break;
		case GE:   R[k][r1] = R[k][r2] >= R[k][r3]; break;
		case SHL:  R[k][r1] = R[k][r2] << R[k][r3]; break;
		case SHR:  R[k][r1] = R[k][r2] >> R[k][r3]; break;
		case MOV:  R[k][r1] = R[k][r2]; break;
		case LDX:  R[k][r1] = M[R[k][r2] + R[k][r3]]; break;
		case STX:  M[R[k][r2] + R[k][r3]] = R[k][r1]; break;
		case RET:  Pc[k] = R[k][r1]; break;
		case TRAP: trap(k,r1,r2); break;
		case PUSH:
			R[k][r1]++;			// increment stack pointer
			M[R[k][r1]] = R[k][r2];
			break;
		case POP:
			R[k][r2] = M[R[k][r1]];
			R[k][r1]--;			// decrement stack pointer
			break;
		case NOT: R[k][r1] = (R[k][r2] == 0) ? ~0 : 0; break;
		case RETI: Pc[k] = R[k][31];	break;
		case PUSHM:				// push r0..15 to stack
			for(i = 0; i < 16; i++){
				R[k][r1]++;		// sp++
				M[R[k][r1]] = R[k][i];
			}
			break;
		case POPM:				// pop stack to r15..r0
			for(i = 15; i >= 0; i--){
				R[k][i] = M[R[k][r1]];
				R[k][r1]--;		// sp--
			}
			break;

		case INT:  interrupt(k,r1); break;
		case CID: R[k][r1] = k; break;
		case WAITX:					// wait
			runflag[k] = 0;
			isWait[k] = 1;
			break;
		case SIGX:
			i = r1;					// core number
			if( isWait[i] ){		// that core is in wait
				runflag[i] = 1;
				isWait[i] = 0;		// clear wait
			}
			break;
		case SYNC: allsync(k); break;
		case EI: intflag[k] = 1; break;
		case DI: intflag[k] = 0; break;
		default:
			error(k,"undefine xop");
		}
		break;
	default:
		error(k,"undefine op");
	}
}
/*
PRIVATE int isEI(int k){	// check trap 16, ei instruction
	return (Ir[k] & 0xFFC00FFF) == 0xFC000013;
}
*/
void sim(void){
	int k;
	T++;
//	printf("T %d\n",T);
	for(k = 0; k < NC; k++){
		if(runflag[k]){
			run(k);  		// do not interrupt just after ei
			checkinterrupt(k);
		}

//		else if(iswfi[k]){
//			if(T - lastInt[i] > DEL){
//				iswfi[i] = 0;
//				runflag[i] = 1;		// clear wfi
//				interrupt(i);
//			}
//		}
	}
}

PRIVATE void initcpu(void){
	int i;
	T = 0;
	for(i = 0; i < NC; i++){
		Pc[i] = 0;
		runflag[i] = 1;
		intflag[i] = 1;		// default enable int
		lastInt[i] = 0;
		isWait[i] = 0;
	}
}

int main(int argc, char *argv[]){
	if( argc < 3 ) {
		printf("usage : sim30 -core objfile\n");
		exit(0);
	}
//	printf("%s\n%s\n",argv[1], argv[2]);
	NC = atoi(argv[1]+1);
	if( NC < 1 ) NC = 1;
	else if( NC > CX ) NC = CX;		// check limit
	loadprogram(argv[2]);
	initcpu();
	interp();
	return 0;
}

