/* as2.c  a simple assembler for s2

format  (case insensitive)

.s                        define symbol
symbol value
 ...
.a address                set address
.c                        code segment
:label opcode operand
 ...
.w 						  data segment
sym sym ...               define words
.e

.s .a .w .c  can occur in any sequence
.e  is the last line of program

operand
L-format: r1 ads
D-format: r1 r2 disp
X-format: r1 r2 r3 xop

addressing mode
absolute: r1 ads
displacement:  r1 @disp r2
index: r1 +r2 r3
register:  r1 r2 r3
immediate: r1 r2 #sym

operand = sym @sym +sym #sym
op:ld st jmp jal jr add sub mul div and or xor shl shr trap

modify from cpu3 assembler A3  version 1.0  7th September 1999
fix bug : undef op, add listing 			2 Oct 1999
add extended instructions  					28 November 2001

Prabhas Chongstitvatana
*/

#include "as2.h"

char 	lbuf[MXBUF];				/* copy of input for error report */
char 	ibuf[MXBUF], *cp = NULL;   	/* input buffer, ch ptr */
static 	char sep[] = " \t\n";      	/* separator char */
char    *w;							/* current input word */
int 	lineno,loc;		   			/* line num, current ads */
FILE 	*fi, *fo, *fl;
token	mem[MXMEM];					/* store tokens for pass2 */
int 	tp = 0;						/* token index */
int 	mark;						/* index to current op token */
int		pass;

/* initial symbol in the symbol table */

typedef struct { char name[8]; int value; } symtype;

symtype	initsym[] = {
	{"LD",LD}, {"ST",ST}, {"JMP",JMP}, {"JAL",JAL}, {"JR",JR},
	{"ADD",ADD}, {"SUB",SUB}, {"MUL",MUL}, {"DIV",DIV},
	{"AND",AND},{"OR",OR},{"XOR",XOR},{"SHL",SHL},{"SHR",SHR},{"TRAP",TRAP},
	{"R0",0},{"R1",1},{"R2",2},{"R3",3},{"R4",4},{"R5",5},
	{"R6",6},{"R7",7},{"R8",8},{"R9",9},{"R10",10},{"R11",11},
	{"R12",12},{"R13",13},{"R14",14},{"R15",15},{"R16",16},{"R17",17},
	{"R18",18},{"R19",19},{"R20",20},{"R21",21},{"R22",22},{"R23",23},
	{"R24",24},{"R25",25},{"R26",26},{"R27",27},{"R28",28},{"R29",29},
	{"R30",30},{"R31",31},
	{"ALWAYS",0},{"EQ",1}, {"NEQ",2},{"LT",3},{"LE",4},{"GE",5},{"GT",6},
	{"STOP",0}, {"PRINT",1}, {"PRINTC",2},
	{"XL",XL}, {"XD",XD}, {"XX",XX}, {"",0}
};

void error(char *s)
{
	if(pass == 1){
		printf("line %d error : %s symbol %s\n%s",lineno,s,w,lbuf);
	}else {
		printf("line %d error: %s token type %d\n",
	    	mem[tp].line,s,mem[tp].type);
	}
	exit(0);
}

#define eqs(a,b)  strcmp((a),(b)) == 0

char *tok()	/* get one token from input */
{
  if( cp != NULL ) cp = strtok(NULL,sep);
  while ( cp == NULL || eqs(cp,";;") ) {
    if( fgets(ibuf,MXBUF,fi) == NULL ) return NULL;
    strcpy(lbuf,ibuf);
	fprintf(fl,"%4d %s",loc,ibuf);
	lineno++;
    cp = strtok(ibuf, sep);
  }
  cp = strupr(cp);   /* convert to uppercase */
// printf("%s ",cp);
  return cp;
}

void store(char type, char mode, int ref){
	if( tp >= MXMEM ) error("out of memory");
	mem[tp].type = type;
	mem[tp].mode = mode;
	mem[tp].ref = ref;
	mem[tp].line = lineno;
	tp++;
}

int installsym(char *s){
	int idx;
	idx = searchsym(s);
	if( idx < 0) idx = putsym(s,UNDEF);
	return idx;
}

void insertsym(char *s, int value){
	putvalue(installsym(s),value);
}

int isnumber(char *s){
	int i = 0;
	if(s[0] == '-') i = 1;
	while( isdigit(s[i]) ) i++;
	if( s[i] == 0 ) return 1;
	return 0;
}

void installtoken(char *s){
	if( isnumber(s) )
		store(NUM,NA,atoi(s));
	else
		store(SYM,NA,installsym(s));
}

int iscmd( char *w )	/* Is token a command ? */
{
  return ( w[0] == '.' );
}

void setop(char mode){			// set mode of op token
	mem[mark].mode = mode;
}

int valueof(char *s, int *type){
	int v;
	*type = NUM;
	if( isnumber(s) ) return atoi(s);
	*type = SYM;
	v = installsym(s);
	return v;
}

void readR(){
	int v,type;
	w = tok();
	v = valueof(w,&type);
	store(type,RR,v);
}

void storew(char *s, int mode){
	int v,type;
	v = valueof(s,&type);
	store(type,mode,v);
	setop(mode);
}

void readM(){
	w = tok();
	switch( w[0] ) {
	case '@': storew(w+1,DISP); readR(); break;
	case '+': storew(w+1,INX); readR(); break;
	case '#': storew(w+1,IMM); break;
	default: storew(w,ABS); break;
	}
}

void readRI(){
	w = tok();
	if( w[0] == '#') storew(w+1,RI);
	else storew(w,RR);
}

void readOP(int mode){
	int v, idx;
	w = tok();
	if(isnumber(w))
		v = atoi(w);
	else {
	    idx = searchsym(w);
		if( idx < 0) error("undefine op");
		v = getvalue(idx);
	}
	store(NUM,mode,v);
	setop(mode);
}

void readcode(int op){
	int v, type;
	mark = tp;
	store(OP,NA,op);
	switch( op ) {
	case LD: readR(); readM(); break;
	case ST: readM(); readR(); break;
	case JMP:
	case JAL:
	case SHL:
	case SHR:
	case TRAP: readR(); readR(); setop(ABS); break;
	case ADD:
	case SUB:
	case MUL:
	case DIV:
	case AND:
	case OR:
	case XOR: readR(); readR(); readRI(); break;
	case JR: readR(); setop(SP); break;
	case XL: readOP(EL); readR(); readR(); break;
	case XD: readOP(ED); readR(); readR(); readR(); break;
	case XX: readOP(EX); readR(); readR(); readR(); readR(); break;
	default: error("undefine op 1");
	}
}

void pass1(){
  int op;
  pass = 1;
  w = tok();
  while( strcmp(w,".E") != 0 ) {
    if( ! iscmd(w)) error("expect .");
    switch( w[1] ) {
    case 'S':			/* scan symbol into table */
      while( !iscmd( w = tok()) )
		putsym(w, atoi(tok()));
      break;
    case 'A' :          /* set address */
      w = tok();
      loc = atoi(w);
	  store(DOTA,NA,loc);
      w = tok();
      break;
    case 'W' :
	  store(DOTW,NA,loc);
      while( !iscmd( w = tok())) {
		if( w[0] == ':' )
			insertsym(w+1, loc);
		else {
			installtoken(w);
			loc++;
      	}
	  }
      break;
    case 'C' :
	  store(DOTC,NA,loc);
      while( !iscmd( w = tok()) ) {
		if( w[0] == ':' )
			insertsym(w+1, loc);
		else {
			op = getsym(w);
			if( op < 0 ) error("undefined op");
			readcode(op);
			loc++;
		}
      } // while case C
    } // switch w[1]
  }  // while
  store(DOTE,NA,loc);
}

int isdot(char type){
	return type == DOTA || type == DOTC || type == DOTW || type == DOTE;
}

/* define constant  opcode encoding
LD 0    ld a 0, ld d 1, ld i 2, ld x 31 9
ST 1	st a 3, st d 4, st x 31 10
JMP 2	jmp 5
JAL 3	jal 6
JR 4	jr 31 11
ADD 5	add i 7, add r 31 0
SUB 6	sub i 8, sub r 31 1
MUL 7	mul i 9, mul r 31 2
DIV 8	div i 10, div r 31 3
AND 9	and i 11, and r 31 4
OR 10	or i 12, or r 31 5
XOR 11	xor i 13, xor r 31 6
SHL 12	shl 31 7
SHR 13	shr 31 8
TRAP 14	trap 31 12
*/

int rdtokval(){
	int v;
	switch( mem[tp].type ){
	case NUM: v = mem[tp].ref; break;
	case SYM: v = getvalue(mem[tp].ref); break;
	default: error("undefine symbol");
	}
	tp++;
	return v;
}

void readarg(int mode, int *a1, int *a2, int *a3, int *a4){
	switch( mode ){
	case SP: *a1 = rdtokval(); break;
	case ABS:
	case IMM:
	case EL: *a1 = rdtokval(); *a2 = rdtokval(); break;
	case INX:
	case DISP:
	case RI:
	case RR:
	case ED: *a1 = rdtokval(); *a2 = rdtokval();
			 *a3 = rdtokval(); break;
	case EX: *a1 = rdtokval(); *a2 = rdtokval();
			 *a3 = rdtokval(); *a4 = rdtokval(); break;
	default: error("unknow addressing mode");
	}
}

void gencode(){
	int op, op2, mode, a1, a2, a3, a4;
	mode = mem[tp].mode;
	op = mem[tp].ref;
	tp++;
	if( op == XL || op == XD || op == XX ) op2 = rdtokval();
	readarg(mode,&a1,&a2,&a3,&a4);
	switch( op ){
	case LD:
		switch( mode ){
		case ABS:  fprintf(fo,"L 0 %d %d\n",a1,a2); break;
		case DISP: fprintf(fo,"D 1 %d %d %d\n",a1,a3,a2); break;
		case INX:  fprintf(fo,"X 31 %d %d %d 9\n",a1,a2,a3); break;
		case IMM:  fprintf(fo,"L 2 %d %d\n",a1,a2); break;
		}
		break;
	case ST:
		switch( mode ){
		case ABS:  fprintf(fo,"L 3 %d %d\n",a2,a1); break;
		case DISP: fprintf(fo,"D 4 %d %d %d\n",a3,a2,a1); break;
		case INX:  fprintf(fo,"X 31 %d %d %d 10\n",a3,a1,a2); break;
		}
		break;
	case JMP: fprintf(fo,"L 5 %d %d\n",a1,a2); break;
	case JAL: fprintf(fo,"L 6 %d %d\n",a1,a2); break;
	case JR:  fprintf(fo,"X 31 %d 0 0 11\n",a1); break;
	case ADD:
	case SUB:
	case MUL:
	case DIV:
	case AND:
	case OR:
	case XOR:
		switch( mode ){
		case RI: fprintf(fo,"D %d %d %d %d\n",op+2,a1,a2,a3); break;
		case RR: fprintf(fo,"X 31 %d %d %d %d\n",a1,a2,a3,op-5); break;
		}
		break;
	case SHL:
	case SHR:  fprintf(fo,"X 31 %d %d 0 %d\n",a1,a2,op-5); break;
	case TRAP: fprintf(fo,"X 31 %d %d 0 12\n",a1,a2); break;
	case XL: fprintf(fo,"L %d %d %d\n",op2,a1,a2); break;
	case XD: fprintf(fo,"D %d %d %d %d\n",op2,a1,a2,a3); break;
	case XX: fprintf(fo,"X %d %d %d %d %d\n",op2,a1,a2,a3,a4); break;
	default: error("undefine op");
	}
}

void pass2(){
	pass = 2;
	tp = 0;
	while( mem[tp].type != DOTE){
		switch(mem[tp].type){
		case DOTA:
			loc = mem[tp].ref;
			fprintf(fo,"a %d\n",loc);
			tp++;
			break;
		case DOTC:
			tp++;
			while(!isdot(mem[tp].type))
				gencode();
			break;
		case DOTW:
			tp++;
			while(!isdot(mem[tp].type))
				fprintf(fo,"w %d\n",rdtokval());
		}
	}
	fprintf(fo,"e\n");
}

void initsymbol(){	/* put reserved words into symbol table */
  int i;
  for(i=0; initsym[i].name[0] != 0; i++)
    putsym(initsym[i].name, initsym[i].value);
}

/* make an obj file name from source */
void makename( char *source, char *obj, char *lis ){
  int n;
  n = strcspn(source,".");
  strncpy(obj,source,n);
  strcpy(obj+n,".obj");
  strncpy(lis,source,n);
  strcpy(lis+n,".lis");
}

//void dumpsymbol();
//void dumptoken();

void main(int argc, char *argv[])
{
  char fout[80], flis[80];

  if( argc < 2 ) {
    printf("usage : as2 inputfile\n");
    exit(0);
  }
  fi = fopen(argv[1],"r");
  if( fi == NULL ){
    printf("input file not found\n");
	exit(0);
  }
  makename(argv[1], fout, flis);
  fl = fopen(flis,"w");
  initsymbol();
  lineno = 0;
  loc = 0;
  pass1();
  fclose(fi);
  fclose(fl);
//  dumpsymbol();
//  dumptoken();
  fo = fopen(fout,"w");
  pass2();
  fclose(fo);
}

/*----------- for debugging

char prmode( int mode){
	switch( mode ){
	case ABS: return 'A';
	case DISP:return '@';
	case INX:return '+';
	case IMM:return '%';
	case RI:return '#';
	case RR:return 'R';
	case SP:return 'S';
	case NA:return '?';
	case EL: return 'L';
	case ED: return 'D';
	case EX: return 'X';
	}
	return 0;
}

void prtoken(int type, int mode, int ref){
	switch( type ){
	case SYM: printf("sym %c %d\n",prmode(mode),ref); break;
	case NUM: printf("num %c %d\n",prmode(mode),ref); break;
	case OP: printf("op %c %d\n",prmode(mode),ref); break;
	case DOTA: printf(".a %d\n",ref); break;
	case DOTC: printf(".c %d\n",ref); break;
	case DOTW: printf(".w %d\n",ref); break;
	case DOTE: printf(".e\n"); break;
	}
}

void dumptoken(){
	int i;
	for(i=0;i<tp;i++)
		prtoken(mem[i].type,mem[i].mode,mem[i].ref);
}

--------------- */
