/*
  ET_SERVO.cpp - Base class that provides Servo Interface With Arduino
 
  ET_SERVO : Library Interface for Arduino 
  -> Modify From Servo.cpp -> Author: Jim Studt, jim@federated.com
  -> Used Timer1 Generate Pulse Control Servo on Digital9,10
  -> Support 2 Servo Connect on Digital-9,Digital-10 Only
  -> Support ATMEGA88/168 Device
  
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 
 Create : 12 November 2008 by Eakachai Makarn.(ETT CO.,LTD. : WWW.ETT.CO.TH)
*/

#include <avr/interrupt.h>
//#include <wiring.h>
#include "Arduino.h"   //For Arduino 1.0
#include <ET_SERVO.h>

int ET_SERVO::attached9  = 0;
int ET_SERVO::attached10 = 0;

//Config Timer1 Generate Pulse Control Servo
void ET_SERVO::seizeTimer1()
{  
  uint8_t oldSREG = SREG;							//Save Register Status
  long double ClockPeriodPWM;
  
  cli();										//Disable Interrupt
  TCCR1A = _BV(WGM11); 								//Fast PWM, ICR1 is Top, OCR1A is Bottom 
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS11);				//Fast PWM, ICR1 is Top,Clock/8 
    
  ClockPeriodPWM = (clockCyclesPerMicrosecond()*(20000L/8));	//20mS Period (Timer1 Clock Prescale = 8) 
  OCR1A = (unsigned int)ClockPeriodPWM;					//"0" = 20mS / 20mS
  OCR1B = (unsigned int)ClockPeriodPWM;					//"0" = 20mS / 20mS
  ICR1  = (unsigned int)ClockPeriodPWM;					//Setup Period of PWM = 20mS
  
  //Reset Timer1 Interrupt Mask of OC1A,OC1B,Overflow
#if defined(__AVR_ATmega328P__) || (__AVR_ATmega168__) || (__AVR_ATmega88__)
  TIMSK1 &=  ~(_BV(OCIE1A) | _BV(OCIE1B) | _BV(TOIE1) );		//Disable Interrupt Mask
#else
  TIMSK &= ~(_BV(TICIE1) | _BV(OCIE1A) | _BV(OCIE1B) | _BV(TOIE1));
#endif
  
  SREG = oldSREG;  								//Restore Register Status    
}

//Release Timer1
void ET_SERVO::releaseTimer1()
{  
  ;
}

ET_SERVO::ET_SERVO() : pin(0), dutycycle(0) 		
{
  dutycycle = 0;									//Default DutyCycle = 0
}

//Connect Servo Pin with Default Pulse Length 1000uS..2000uS
int ET_SERVO::attach(int servo_pin)
{
  attach(servo_pin, 1000, 2000);
  return 1;
}

//Connect Servo Pin with Pulse Length min..max
int ET_SERVO::attach(int servo_pin, unsigned int min_pulse, unsigned int max_pulse)	
{
  long double ClockDutyCycle;

  if (servo_pin != 9 && servo_pin != 10) return 0;			//Error if Pin Not Digital-9,Digital-10
  
  minpulse = min_pulse;								//Minimum Pulse Update
  maxpulse = max_pulse;								//Maximum Pulse Update

  pin = servo_pin;								//Connect Pin Servo	

  digitalWrite(pin, LOW);							//Default Pulse = Low
  pinMode(pin, OUTPUT);

  if (!attached9 && !attached10) seizeTimer1();				//If New Connect = Start Timer1

  if(dutycycle>0)
  {
    ClockDutyCycle = (clockCyclesPerMicrosecond()*(dutycycle/8));  
  }
  else
  {
    ClockDutyCycle = (clockCyclesPerMicrosecond()*(20000L/8));  
  }
  
  if (pin == 9) 
  {
    attached9 = 1;
    TCCR1A = (TCCR1A & ~_BV(COM1A0)) | _BV(COM1A1);			//Toggle OC1A on Compare Match
    OCR1A = (unsigned int)ClockDutyCycle;
  }
  
  if (pin == 10) 
  {
    attached10 = 1;
    TCCR1A = (TCCR1A & ~_BV(COM1B0)) | _BV(COM1B1);			//Toggle OC1B on Compare Match
    OCR1B = (unsigned int)ClockDutyCycle;
  }

  return 1;										//Connect Servo Pin Ready
}

//Disconnect Servo Pin
void ET_SERVO::detach()
{
  long double ClockDutyCycle;

  ClockDutyCycle = (clockCyclesPerMicrosecond()*(20000L/8));  

  if (pin == 9) 									//Disconnect Digital-9 From Servo
  {
    attached9 = 0;
    TCCR1A = TCCR1A & ~_BV(COM1A0) & ~_BV(COM1A1);			//Normal Port Operation(OC1A Disconnect From Timer1)
    pinMode(pin, INPUT);
    dutycycle = 0;								//Default Duty Cycle
    OCR1A = (unsigned int)ClockDutyCycle;
  } 
  
  if (pin == 10) 									//Disconnect Digital-10 From Servo
  {
    attached10 = 0;
    TCCR1A = TCCR1A & ~_BV(COM1B0) & ~_BV(COM1B1);			//Normal Port Operation(OC1B Disconnect From Timer1)
    pinMode(pin, INPUT);
    dutycycle = 0;								//Default Duty Cycle
    OCR1B = (unsigned int)ClockDutyCycle;
  }

  if (!attached9 && !attached10) releaseTimer1();	
}

//Adjust Position Servo
void ET_SERVO::updateDutyCycle(unsigned int position)
{
  long double ClockDutyCycle;

  if (position < minpulse) position = minpulse;
  if (position > maxpulse) position = maxpulse;
  dutycycle = position;

  ClockDutyCycle = (clockCyclesPerMicrosecond()*(position/8));  
  if (pin == 9)  OCR1A = (unsigned int)ClockDutyCycle;
  if (pin == 10) OCR1B = (unsigned int)ClockDutyCycle;
}

//Read Position Servo
unsigned int ET_SERVO::readDutyCycle()
{
  return dutycycle;								//Return DutyCycle of PWM
}

//Verify Pin Servo Connect
int ET_SERVO::attached()
{
  if (pin == 9  && attached9)  return 1;					//Connect on Digital-9
  if (pin == 10 && attached10) return 1;					//Connect in Digital-10
  return 0;										//Disconnect
}

void ET_SERVO::setMinPulse(unsigned int min_pulse)         		//Min Pulse length in microseconds(Default 1000uS:1mS)
{
  minpulse = min_pulse;								//Minimum Pulse Update
}

void ET_SERVO::setMaxPulse(unsigned int max_pulse)         		//Max Pulse length in microseconds(Default 2000uS:2mS)
{
  maxpulse = max_pulse;								//Maximum Pulse Update
}

//Forward Modify Servo("1" = 2mS / 20mS)
void ET_SERVO::modifyServoForward()						//Forward  Modify  Servo(1mS)
{
  long double ClockDutyCycle;

  ClockDutyCycle = (clockCyclesPerMicrosecond()*(2000L/8));  	//2mS Pulse
  if (pin == 9)  OCR1A = (unsigned int)ClockDutyCycle;
  if (pin == 10) OCR1B = (unsigned int)ClockDutyCycle;

  dutycycle = 2000;								//Duty Cycle = 2000uS
}

//Backward Modify Servo("1" = 1mS / 20mS)
void ET_SERVO::modifyServoBackward()					//Backward Modify Servo(2mS)
{
  long double ClockDutyCycle;

  ClockDutyCycle = (clockCyclesPerMicrosecond()*(1000L/8));  	//1mS Pulse
  if (pin == 9)  OCR1A = (unsigned int)ClockDutyCycle;
  if (pin == 10) OCR1B = (unsigned int)ClockDutyCycle;

  dutycycle = 1000;								//Duty Cycle = 1000uS
}

//Stop Modify Servo ("0" = 20mS / 20mS)
void ET_SERVO::modifyServoStop()	      				//Stop Modify Servo("0")
{
  long double ClockDutyCycle;

  ClockDutyCycle = (clockCyclesPerMicrosecond()*(20000L/8));  	//20mS Pulse
  if (pin == 9)  OCR1A = (unsigned int)ClockDutyCycle;
  if (pin == 10) OCR1B = (unsigned int)ClockDutyCycle;

  dutycycle = 0;									//Duty Cycle = 0uS
}
