/* nut.c
   for teaching digital systems  2004
   base on base2.c				21 Nov 2004
   add global var "let"			 4 Jan 2005

   P. Chongstitvatana
   nut3
   	no macro
	add enum, string			31 Oct 2005
   clean up for teaching 2006	15 June 2006
*/

#include "nut.h"

symrec symtab[MAXNAMES];	// symbol table
int numNames = 0;			// number of symbols

struct {
	namestr key;
	int type;
	int value;
} keynames[] = {			// reserved words
	{"if",tyOP,xIF},
	{"while",tyOP,xWHILE},
	{"set",tyOPX,ySET},
	{"setv",tyOPX,ySETV},
	{"do",tyOP,xDO},
	{"new",tyOP,xNEW},
	{"+",tyOP,xADD},
	{"-",tyOP,xSUB},
	{"=",tyOP,xEQ},
	{"*",tyOP,xMUL},
	{"/",tyOP,xDIV},
	{"<",tyOP,xLT},
	{">",tyOP,xGT},
	{"vec",tyOPX,yVEC},
	{"sys",tySYS,xSYS},
	{"&",tyOP,xBAND},
	{">>",tyOP,xSHR},
	{"<<",tyOP,xSHL},
	{"",0,0}
};

int line;				// current line
char *cp;				// current char
char inbuf[MAXFIN];		// input buffer
char token[80];			// current token
int numLocal;			// number of local var

void error( char *mess ){
	printf("error: %s at line %d\n", mess,line);
	exit(0);
}

// --------- symbol table ---------

// search if not found insert return index
int install(namestr nm){
	int i;
	for(i=0; i<numNames; i++)   // sequential search !
		if( eqs(symtab[i].name, nm) ) break;
	if( i == numNames){ 		// not found
		if( i >= MAXNAMES )
			error("symtab overflow");
		strcpy(symtab[i].name, nm);
		symtab[i].type = tyUD;	// undef
		numNames++;
	}
	return i;
}

// initialise reserved words in symbol table
void initNames(void){
	int i, idx;
	i = 0;
	while( keynames[i].key[0] != 0 ) {
		idx = install(keynames[i].key);
		symtab[idx].type = keynames[i].type;
		symtab[idx].val = keynames[i].value;
		i++;
	}
}

void dumpsymtab(FILE *fo){		// start at 16
	int i, t, n;
	for(n=0, i=16; i<numNames; i++){	// count no. of fun,gvar
		t = symtab[i].type;
		if( t == tyFUN || t == tyGVAR )
			n++;
	}
	fprintf(fo,"%d\n",n);
	for(i=16; i<numNames; i++){
		t = symtab[i].type;
		if( t == tyFUN || t == tyGVAR ){
			fprintf(fo,"%d %s %d %d %d %d\n",
			i, symtab[i].name, t,
			symtab[i].val, symtab[i].arity, symtab[i].lv);
		}
	}
}

// -------- lexical analyser ---------

// read from stdin, redirect from a file
void readinfile(void) {
	int n;
/*
	fp = fopen(fname,"rt");
	if( fp == NULL ) {
		printf("cannot open : %s\n",fname);
		exit(-1);
	}
	n = fread(inbuf,1,MAXFIN-2,fp);
	fclose(fp);
*/
	n = fread(inbuf,1,MAXFIN-2,stdin);
	inbuf[n] = EOF_CHAR;
	inbuf[n+1] = 0;
	cp = inbuf;
	line = 1;
}

int isSpecial(char c){
	return c == '(' || c == ')' || c == EOF;
}

int isBlank(char c){
	return c == 32 || c == '\t' || c == '\n' || c == ';';
}

// skip " \t\n;" and comment, counting line
char *skipblank(char *s){
	while( isBlank(*s)){
		if( *s == '\n') line++;
		if( *s == ';'){
			while( *s != '\n') s++;	// skip to eol
			line++;
		}
		s++;
	}
	return s;
}

// tok blank c<- c2+1 -> c+1+c2
//   ^---------^-----------^
//   leave pointer to next to the end of current token

void tokenise(void){
	int c2;
	cp = skipblank(cp);
	if( isSpecial(*cp)) {
		token[0] = *cp;
		token[1] = 0;
		cp++;
	}else if( *cp == '"'){			// string
		c2 = strcspn(cp+1,"\"");
		strncpy(token,cp,c2+1);
		token[c2+1] = 0;
		cp += c2+2;
	}else{							// name
		c2 = strcspn(cp+1,DELIM);
		strncpy(token,cp,c2+1);
		token[c2+1] = 0;
		cp += c2+1;
	}
//	printf("%s ",token);
}

int isNumber(namestr nm){
	int i = 0;
	if(nm[0] == '-') i = 1;
	while( isdigit(nm[i]) ) i++;
	if( nm[i] == 0 ) return 1;
	return 0;
}

int isString(char *s){		// start with "
	return *s == '"';
}

// ------ constructor --------

int isAtom(exp e){ return (e & 0x80000000) != 0;}

exp cons(exp e, exp list){
	exp h;
	h = (exp)new(2);
	sethead(h,e);
	settail(h,list);
	return h;
}

// ------ parser --------------

void expect(char *s){
	char mess[80];
	if( ! eqs(token,s) ){
		sprintf(mess,"expect %s",s);
		error(mess);
	}
}

/* return atom */
exp parseName(void){
	int idx, v, v2;
	idx = install(token);
	v = symtab[idx].val;
	switch( symtab[idx].type ){
	case tyOP:  return mkATOM(v,0);
	case tyVAR: return mkATOM(xGET,v);
	case tyGVAR: return mkATOM(xLD,v);
	case tyFUN:	return mkATOM(xCALL,idx);
	case tyOPX:
		tokenise();	// get var-name
		idx = install(token);
		v2 = symtab[idx].val;
		if(symtab[idx].type == tyVAR){
			switch(v){
			case ySET: return mkATOM(xPUT,v2);
			case ySETV: return mkATOM(xSTX,v2);
			case yVEC: return mkATOM(xLDX,v2);
			default: error("unknow op");
			}
		}else{		// tyGVAR
			switch(v){
			case ySET: return mkATOM(xST,v2);
			case ySETV: return mkATOM(xSTY,v2);
			case yVEC: return mkATOM(xLDY,v2);
			default: error("unknow op");
			}
		}
	case tySYS:
		tokenise();		// get sys number
		return mkATOM(xSYS,atoi(token));
	case tyENUM:
		return mkATOM(xLIT,v);
	}
	return 0;		// never come here
}

exp parseEL(void){
	exp e, el;
	tokenise();
	if( eqs(token,")") ) return NIL;
	e = parseExp();
	el = parseEL();
	return cons(e, el);
}

void decode(exp a, int *op, int *arg){
	*op = (a >> 24) & 0x7f;
	*arg = a & 0x0ffffff;
}

/* initial symbol is already in token[] */
exp parseExp(void){
	exp nm, el;
	if( eqs(token,"(") ) {		// it is a list
		tokenise();
		nm = parseName();
		el = parseEL();			// normal arg list
		return cons(nm, el);
	}							// else it is an atom
	if( isNumber(token) )		// it is a number
		return mkATOM(xLIT, atoi(token));
	if( isString(token) )
		return mkATOM(xSTR, mkSTR(token+1));
	return parseName();			// else OP, OPX, VAR, FUN
}

void installLocal(namestr nm){
	int idx;
	numLocal++;
	idx = install(nm);
	symtab[idx].type = tyVAR;
	symtab[idx].val = numLocal;
}

// NL can be either singleton or list
void parseNL(void){
	tokenise();
	if( eqs(token,"(") ){
		tokenise();
		while( ! eqs(token,")") ){
			installLocal(token);
			tokenise();
		}
	}else
		installLocal(token);
}

// parse def return index into symtab
int parseDef(void){
	int idx, arity, k, op, lv;
	exp e;
	tokenise();  	// get fun name
	idx = install(token);
	symtab[idx].type = tyFUN;
	numLocal = 0;
	parseNL();
	arity = numLocal;
	parseNL();
	tokenise();
	e = parseExp();
	tokenise();		// skip ")"
	if(isAtom(e))
		e = cons(e,NIL); // body must be list
	symtab[idx].arity = arity;
	symtab[idx].lv = numLocal;
	k = (arity << 8) | (numLocal & 0x0ff);
	symtab[idx].val = cons(mkATOM(xFUN,k),cons(e,NIL));
	return idx;
}

void prName(int idx){
	printf("%s\n",symtab[idx].name);
}

// define global var, alloc data
void parseLet(void){
	int idx;
	tokenise();
	while( !eqs(token,")")){
		idx = install(token);
		if(symtab[idx].type != tyUD)
			error("redefine global var");
		symtab[idx].type = tyGVAR;
		symtab[idx].val = newdata();	// alloc mem
		printf("%s\n",token);
		tokenise();
	}
}

void parseEnum(void){
	int k, idx;
	tokenise();
	if(!isNumber(token)) error("expect number");
	k = atoi(token);
	tokenise();
	while( !eqs(token,")")){
		idx = install(token);
		if(symtab[idx].type != tyUD)
			error("redefine enum name");
		symtab[idx].type = tyENUM;
		symtab[idx].val = k;
		k++;
		tokenise();
	}
}

void parse(void){
	int idx;
	tokenise();
	while(token[0] != EOF_CHAR){
		expect("(");
		tokenise();
		if( eqs(token,"def") ){
			idx = parseDef();
			prName(idx);
			prList(symtab[idx].val);
			nl();
	    }else if( eqs(token,"let") ){
			parseLet();
		}else if( eqs(token,"enum") ){
			parseEnum();
		}else
			error("unknown keyword");
		tokenise();
	}
}

// ------- end parser --------

int LV;

void reAtom(exp e){
	int op, arg;
	decode(head(e),&op,&arg);
	switch(op){
	case xFUN: LV = arg & 255; break;
	case xGET:
	case xPUT:
	case xLDX:
	case xSTX:
		sethead(e,mkATOM(op,LV-arg+1));
		break;
	case xCALL:
		sethead(e,mkATOM(op,symtab[arg].val));
		break;
	}
}

// resolve N-code at a1..a2
// set ads of call, rename local var 1..n to n..1
void reName( exp e){
	if( e == NIL ) return;
	if( isAtom(head(e)) ){
		reAtom(e);
		reName(tail(e));
	}else{
		reName(head(e));
		reName(tail(e));
	}
}

// checking any undefined function call
void check(int a1, int a2){
	int i, h, lv, op, arg;
	for(i=a1; i<a2; i+=2){
		h = heap[i];
		if( isAtom(h) ){
			decode(h,&op,&arg);
			if(op == xCALL && arg == 0)
				printf("call 0 at %d\n",i);
		}
	}
}

// instantiate call and rename local var
void resolve(void){
	int i;
	for(i=16; i<numNames; i++)
		if(symtab[i].type == tyFUN)
			reName(symtab[i].val);
}

int main(void){
	initNames();
	readinfile();
	parse();
	resolve();
//	dumpsymtab();
//	check(2,heapFree);
	outobj("a.obj");
	return 0;
}

